// $Id: CCriticalSection.cpp,v 1.5 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CCriticalSection.hpp"
using Exponent::Threading::CCriticalSection;

#ifdef WIN32
	// Nothing to include
#else
	#include <unistd.h>
	#include <pthread.h>
#endif

#include <time.h>

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CCriticalSection, CCountedObject);

//	===========================================================================
CCriticalSection::CCriticalSection()
				: m_criticalSection(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CCriticalSection);
#ifdef WIN32
	// Create the critical section
	m_criticalSection = new CRITICAL_SECTION;

	// Initialise the critical section
	InitializeCriticalSection(m_criticalSection);
#else
	// Create the critical section
	MPCreateCriticalRegion(&m_criticalSection);
#endif
}

//	===========================================================================
CCriticalSection::~CCriticalSection()
{
	EXPONENT_CLASS_DESTRUCTION(CCriticalSection);
#ifdef WIN32
	// Remove the critical section from the windows list
	DeleteCriticalSection(m_criticalSection);

	// Delete the actual pointer
	delete m_criticalSection;
#else
	// Free the critical section from the system list
	MPDeleteCriticalRegion(m_criticalSection);
#endif
}

//	===========================================================================
bool CCriticalSection::tryToEnterCriticalSection(const long spinTimeInMilliseconds) const
{
#ifdef WIN32
	// Store the wait time in a writable variable
	long waitTime = spinTimeInMilliseconds;

	// We havent obtained yet..
	bool obtained = false;

	// If we have a critical section
	if (m_criticalSection)
	{
		// And the wait time is greater than zero
        if (waitTime > 0)
		{
            while (waitTime > 0 && !obtained)
			{
				// Try to get it
                obtained = (TryEnterCriticalSection(m_criticalSection) > 0);

				// If we failed to get it
                if (!obtained)
				{
					// Pause for 5 ms
                    Sleep(5);

					// Reduce the wait time
                    waitTime -=5;
                }
            }
        }
        else
		{
			// We directly enter the critical section
            EnterCriticalSection(m_criticalSection);

			// We have got our critical section
            obtained = true;
        }
	}
	return obtained;
#else
	return (MPEnterCriticalRegion(m_criticalSection, spinTimeInMilliseconds * kDurationMillisecond) == noErr);
#endif
}

//	===========================================================================
void CCriticalSection::enterCriticalSection(void) const
{
#ifdef WIN32
    if (m_criticalSection)
	{
        EnterCriticalSection(m_criticalSection);
	}
#else
	MPEnterCriticalRegion(m_criticalSection, kDurationForever);
#endif
}

//	===========================================================================
void CCriticalSection::leaveCriticalSection(void) const
{
#ifdef WIN32
	if (m_criticalSection)
	{
        LeaveCriticalSection(m_criticalSection);
	}
#else
	MPExitCriticalRegion(m_criticalSection);
#endif
}